/**
* \file: handle_umount_request_cmd.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <stdbool.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <poll.h>
#include <errno.h>

#include "handle_mnt_umnt_request_cmd/handle_umount_request_cmd.h"
#include "utils/path.h"
#include "automounter_types.h"
#include "automounter_api.h"
#include "automounter_api_ctrl.h"

//------------------------------------------ private attributes -------------------------------------------------------
typedef enum identifier_type_t
{
	DEVICE_ID,
	PARTITION_ID,
	MOUNTPOINT
} identifier_type_t;

static char* identifier=NULL;
static identifier_type_t identifier_type=MOUNTPOINT;
static bool waiting_for_request_done_signal=true;
static error_code_t request_result=RESULT_OK;
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------------ private function declaration ---------------------------------------------
static error_code_t handle_umount_cmd_parse_args(int argc, char *argv[]);
static error_code_t handle_umount_cmd_start(void);
static void handle_umount_cmd_print_help(void);
static void handle_umount_cmd_print_usage(void);

static error_code_t handle_umount_cmd_init(void);
static void handle_umount_cmd_deinit(void);

static void handle_umount_done_callback(int request_id, error_code_t result, const char *error_message);
static error_code_t handle_umount_cmd_mainloop(void);

//---------------------------------------------------------------------------------------------------------------------

//------------------------------------------ vtable for the cmd -------------------------------------------------------
automounterctl_command_vtable_t handle_umount_request_cmd_vtable =
{
		.command_description =
				"Triggers the automounter to umount a partition or all partitions of a device.",
				.command = "umount",
				.init = handle_umount_cmd_init,
				.parse_args = handle_umount_cmd_parse_args,
				.start = handle_umount_cmd_start,
				.deinit = handle_umount_cmd_deinit
};
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------------ private function definition ----------------------------------------------
static error_code_t handle_umount_cmd_parse_args(int argc, char *argv[])
{
	//no additional parameter given, we are missing something
	if (argc == 2)
	{
		printf("Too few arguments. A partition or device to be unmounted must be passed.\n");
		handle_umount_cmd_print_usage();
		return RESULT_HELP_PRINTED;
	}

	//we can be sure that we have more than 2 parameters (automounterctl <command>) passed,
	//otherwise we would have never got here.
	if (strcmp(argv[2], "--help") == 0 || strcmp(argv[2], "-h") == 0)
	{
		handle_umount_cmd_print_help();
		return RESULT_HELP_PRINTED;
	}
	else if (strcmp(argv[2], "--device") == 0 || strcmp(argv[2], "-d") == 0)
		identifier_type=DEVICE_ID;
	else if (strcmp(argv[2], "--partition") == 0 || strcmp(argv[2], "-p") == 0)
		identifier_type=PARTITION_ID;
	else if (strcmp(argv[2], "--mountpoint") == 0 || strcmp(argv[2], "-m") == 0)
		identifier_type=MOUNTPOINT;
	else
	{
		printf("Unknown argument: %s.\n", argv[2]);
		handle_umount_cmd_print_usage();
		return RESULT_HELP_PRINTED;
	}

	if (argc < 4)
	{
		if (identifier_type==MOUNTPOINT)
			printf("Too few arguments. A mount point needs to be passed.\n");
		else
			printf("Too few arguments. An id needs to be passed.\n");
		handle_umount_cmd_print_usage();
		return RESULT_HELP_PRINTED;
	}

	identifier = argv[3];

	//in case remove the trailing slash
	path_remove_trailing_slash(identifier);

	return RESULT_OK;
}

static void handle_umount_cmd_print_help(void)
{
	printf("\n");
	printf("Automounter Control Utility - Used to control the ADIT automounter daemon.\n\n");
	printf("automounterctl %s --device <id> | --partition <id> | --mountpoint <mp> | --help\n",
			handle_umount_request_cmd_vtable.command);
	printf("\t-h,--help:\t\tdisplays this help and exits.\n");
	printf("\t-d,--device:\t\tunmounts the device with given <id>\n");
	printf("\t-p,--partition:\t\tunmounts the partition with given <id>\n");
	printf("\t-m,--mountpoint:\tunmounts the partition mounted at given mount point <mp>\n\n");
	printf("This command triggers the automounter daemon to umount a given partition or \n"
			"all partitions of  given device. The partition or device is defined by an id\n"
			"given as command line parameter. Partitions or device that have not been mounted\n"
			"by the automounter before are not touched.\n\n");
}

static void handle_umount_cmd_print_usage(void)
{
	printf("\n");
	printf("Usage: automounterctl %s --device <id> | --partition <id> | --mountpoint <mp> | --help\n",
			handle_umount_request_cmd_vtable.command);
	printf("\t-h,--help:\t\tdisplays this help and exits.\n");
	printf("\t-d,--device:\t\tunmounts the device with given <id>\n");
	printf("\t-p,--partition:\t\tunmounts the partition with given <id>\n");
	printf("\t-m,--mountpoint:\tunmounts the partition mounted at given mount point <mp>\n\n");
}

static error_code_t handle_umount_cmd_start(void)
{
	error_code_t result;
	result=automounter_api_connect();

	waiting_for_request_done_signal=true;
	request_result=RESULT_OK;

	if (result == RESULT_OK)
	{
		if (identifier_type==DEVICE_ID)
			result = automounter_api_umount_device(identifier,-1,handle_umount_done_callback);
		else if (identifier_type==PARTITION_ID)
			result = automounter_api_umount_partition_by_id(identifier,-1,handle_umount_done_callback);
		else // identifier==MOUNTPOINT
			result = automounter_api_umount_partition_by_mountpoint(identifier,-1,handle_umount_done_callback);
	}

	if (result != RESULT_OK)
	{
		printf("Error executing the remount command.\n");
		return result;
	}

	//we sent the command successfully, now we are waiting for a response
	while(result == RESULT_OK && waiting_for_request_done_signal)
		result=handle_umount_cmd_mainloop();

	//check if we had an error causing us to leave the mainloop
	if (result!=RESULT_OK)
	{
		printf("Connection to the automounter got lost while processing the request.\n");
		return result;
	}

	// we got an answer. Check the results transmitted from the automounter
	if (request_result==RESULT_NOT_MOUNTED)
	{
		if (identifier_type==DEVICE_ID)
			printf("The device has been unmounted before.\n");
		else
			printf("The partition has been unmounted before.\n");
	}
	else if (request_result==RESULT_INVALID)
	{
		if (identifier_type==MOUNTPOINT)
			printf("Given mount point not found or not under the responsibility of the automounter.\n");
		else
			printf("The given id does not belong to a device or partition currently under the responsibility"
					" of the automounter.\n");
	}
	else if (request_result==RESULT_NORESOURCE)
		printf("We run into resource issues. Memory problems?\n");

	return request_result;
}

static error_code_t handle_umount_cmd_init(void)
{
	return automounter_api_init("automounterctl", LOGGER_LEVEL_ERROR, true);
}

static void handle_umount_cmd_deinit(void)
{
	automounter_api_disconnect();
	automounter_api_deinit();
}

static void handle_umount_done_callback(int request_id, error_code_t result, const char *error_message)
{
	//we passed -1 here because we are not sending more than one request at a time. So we know from which
	//request the answer comes.
	(void)request_id;
	//we are extracting the message from the result code for now
	(void)error_message;
	waiting_for_request_done_signal=false;
	request_result=result;
}

static error_code_t handle_umount_cmd_mainloop(void)
{
	error_code_t result=RESULT_OK;
	int am_pollfd;
	struct pollfd am_pollfd_struc;

	am_pollfd=automounter_api_get_pollfd();
	if (am_pollfd==-1)
		return RESULT_INVALID;

	am_pollfd_struc.fd=am_pollfd;
	am_pollfd_struc.events=POLLIN;

	//we sent the command successfully, now we are waiting for a response
	while(result == RESULT_OK && waiting_for_request_done_signal)
	{
		if (poll(&am_pollfd_struc,1,-1)<=0)
		{
			if (errno!=EINTR)
			{
				printf("ERROR: Automounterctl detected problems with its poll file descriptor in the main loop.\n");
				result=RESULT_NORESOURCE;
			}
			else
				waiting_for_request_done_signal=false;
			continue;
		}
		automounter_api_dispatch_event();
	}

	return result;
}
//---------------------------------------------------------------------------------------------------------------------
